//===============================================================================
// Microsoft patterns & practices Enterprise Library
// Configuration Application Block
//===============================================================================
// Copyright  Microsoft Corporation.  All rights reserved.
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY
// OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT
// LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
// FITNESS FOR A PARTICULAR PURPOSE.
//===============================================================================

using System;
using System.Reflection;
using Microsoft.Practices.EnterpriseLibrary.Configuration.Factory;
using Microsoft.Practices.EnterpriseLibrary.Configuration.Properties;

namespace Microsoft.Practices.EnterpriseLibrary.Configuration.Injection
{
	/// <summary>
	/// Represents the base class from which all implementations of property parameter attributes must derive.
	/// </summary>
	/// <remarks>
	/// This attribute uses the property name to get the value through reflection for the paramter value.
	/// </remarks>
    public class PropertyParameterAttribute : ConstructorParameterAttribute
	{
		private readonly string propertyName;

		/// <summary>
		/// Initialize a new instance of the <see cref="PropertyParameterAttribute"/> class with the order, the parameter type, and the property name.
		/// </summary>
		/// <param name="order">The ordinal position of the constructor parameter.</param>
		/// <param name="parameterType">The parameter type.</param>
		/// <param name="propertyName">The name of the property to get the parameter value.</param>
		public PropertyParameterAttribute(int order, Type parameterType, string propertyName)
			: base(order, parameterType)
		{
			if (string.IsNullOrEmpty(propertyName))
			{
				throw new ArgumentException(Resources.ExceptionStringNullOrEmpty, "propertyName");
			}
			this.propertyName = propertyName;
		}

		/// <summary>
		/// Gets the property name that contains the value for the parameter.
		/// </summary>
		/// <value>
		/// The property name that contains the value for the parameter.
		/// </value>
		public string PropertyName
		{
			get { return propertyName; }
		}

		/// <summary>
		/// Gets the <see cref="PropertyParameterFactory"/> used to create the parameter value.
		/// </summary>
		/// <param name="objectConfiguration">The object configuration used to retrieve the value.</param>
        /// <returns>A <see cref="PropertyParameterFactory"/> object.</returns>
        protected internal override IObjectFactory<TObjectToCreate, TObjectConfiguration> GetParameterFactory<TObjectToCreate, TObjectConfiguration>(TObjectConfiguration objectConfiguration)
		{
			PropertyInfo propertyInfo = objectConfiguration.GetType().GetProperty(propertyName);
			if (propertyInfo == null)
			{
				throw new InvalidOperationException(string.Format(Resources.Culture, Resources.ExceptionSourcePropertyDoesNotExist, propertyName, objectConfiguration.GetType().Name));
			}

            object innerFactory = GetInnerFactoryGeneric(typeof(TObjectToCreate), propertyInfo, objectConfiguration);

            return (IObjectFactory<TObjectToCreate, TObjectConfiguration>)CreatePropertyParameterFactory<TObjectToCreate, TObjectConfiguration>(propertyInfo, innerFactory);
		}

        private object CreatePropertyParameterFactory<TObjectToCreate, TObjectConfiguration>(PropertyInfo propertyInfo, object innerFactory)
        {
            Type propertyParameterFactory = typeof(PropertyParameterFactory<,,>);
            propertyParameterFactory = propertyParameterFactory.MakeGenericType(typeof(TObjectToCreate), propertyInfo.PropertyType, typeof(TObjectConfiguration));
            return Activator.CreateInstance(propertyParameterFactory, new object[] { propertyInfo, innerFactory });
        }

        private object GetInnerFactoryGeneric(Type objectToCreateType, PropertyInfo propertyInfo, object objectConfiguration)
        {
            MethodInfo getInnerFactory = typeof(PropertyParameterAttribute).GetMethod("GetInnerFactory", BindingFlags.NonPublic | BindingFlags.Instance);
            getInnerFactory = getInnerFactory.MakeGenericMethod(objectToCreateType, propertyInfo.PropertyType);

            object innerFactory = getInnerFactory.Invoke(this, new object[] { propertyInfo, propertyInfo.GetValue(objectConfiguration, new object[0]) });

            return innerFactory;
        }

		/// <summary>
		/// When overriden by a class, gets the <see cref="TransformationStrategy"/> to use to transform the property value to the value for the parameter.
		/// </summary>
		/// <returns>The <see cref="TransformationStrategy"/> to use to transform the property value to the value for the parameter.</returns>
        protected internal virtual IObjectFactory<TObjectToCreate, TObjectConfiguration> GetInnerFactory<TObjectToCreate, TObjectConfiguration>(PropertyInfo propertyInfo, TObjectConfiguration objectConfiguration)
        {
            ConverterBaseAttribute[] attributes = (ConverterBaseAttribute[])propertyInfo.GetCustomAttributes(typeof(ConverterBaseAttribute), false);

            if (attributes.Length == 1)
            {
                return attributes[0].GetFactory<TObjectToCreate, TObjectConfiguration>(objectConfiguration);
            }
            else
            {
                return new NullFactory<TObjectToCreate, TObjectConfiguration>();
            }
        }
	}
}
